// LDA64_EthTest.c : This file contains the 'main' function. Program execution begins and ends there.
//
// This version is an example CLI for Linux
#include <stdbool.h>
#include <stdio.h>
#include <stdint.h>         
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include "ldadrvr.h"

#define FALSE 0
#define TRUE !FALSE

const char* ldadevices[] = {"192.168.100.5"};
#define DEBUG_LEVEL 1
#define LDA_DEVICES  (sizeof(ldadevices)/sizeof(ldadevices[0]))

// IP Address Validator
#define DELIM "."

// ------------------------------ - Equates-----------------------------------------------
#define CL_SWP_DIRECTION		0x04	// MASK: bit = 0 for ramp up, 1 for ramp down 
#define CL_SWP_CONTINUOUS		0x02	// MASK: bit = 1 for continuous ramping
#define CL_SWP_ONCE				0x01	// MASK: bit = 1 for single ramp
#define CL_SWP_BIDIRECTIONALLY	0x10	// MASK: bit = 1 for bidirectional ramps (V2 LDA's only)

// --------------------------- Variables -------------------------------------------------
static char *ndeviceip;
static int IdleTime = 1;		// default idle time is 1 ms
static int HoldTime = 1;		// default hold time is 1 ms
static float AStart = 0;		// default atten start level is 0 db.
static float AStop = 252;		// default atten stop, for most devices this is 63 db so we use that.
static int Dwell = 1000;		// default dwell time is 1 second for first ramp phase
static int Dwell2 = 1000;		// default dwell time is 1 second for second ramp phase (V2 LDA's only)
static float AStep = 2;			// default step size is .5db, some LDA's have larger minimum steps
static float AStep2 = 2;		// default second phase step size for LDA's that support bidirectional ramps

static int WorkingFrequency = 0;	// working frequency for the HiRes attenuators
static float Attenuation = 0;		// default attenuation is 0db, entered as a floating point value

static int ScaledAttenuation = 0;	// temporary storage for scaled attenuation values
static int SerialNumber = 0;		// used to hold the serial number for the get serial number command

static int RFOnOff = 1;				// really used as a bool -- if non zero, turn on the RF output

static int Sweep_mode = 0;			// used to control the sweep mode
static int GetParam = 0;			// the low byte is the GET command byte

static int ProfileIndex = 0;		// the element in the profile we want to set
static int ProfileLength = 0;		// the length of the profile
static int ProfileValue = 0;		// the profile element's value

static int Channel = 1;				// we just default to the first channel
static int NumToRamp = 0;			// number of channels we should ramp (also used as an index -- post increment!)
static int RampMode[64];
static int iRampMode = 0;
static char gldadevicesip[16];       // devices ip address

bool gbDeviceOpen = FALSE;
bool gbWantSetIdle = FALSE;
bool gbWantSetHold = FALSE;
bool gbWantSetAStart = FALSE;
bool gbWantSetAStop = FALSE;
bool gbWantSetDwell = FALSE;
bool gbWantSetDwell2 = FALSE;
bool gbWantStartSweep = FALSE;
bool gbWantSetAStep = FALSE;
bool gbWantSetAStep2 = FALSE;
bool gbWantSetWorkingFrequency = FALSE;
bool gbWantSetAttenuation = FALSE;
bool gbWantSaveSettings = FALSE;
bool gbWantGetParam = FALSE;
bool gbBatchMode = FALSE;
bool gbWantSetRFOnOff = FALSE;
bool gbQuietMode = FALSE;
bool gbWantSetProfileElement = FALSE;
bool gbWantSetProfileLength = FALSE;
bool gbWantChannel = FALSE;
bool gbWantMCRamp = FALSE;


//***************************************************************************
//
// Validate netconfig digits and return 1 if string contain only digits, else return 0
//
//*****************************************************************************
static int valid_digit(char *ip_str){
    while (*ip_str) {
        if (*ip_str >= '0' && *ip_str <= '9')
            ++ip_str;
        else
            return 0;
    }
    return 1;
}

//***************************************************************************
//
// Validate IP Configuration -return 1 if IP string is valid, else return 0
//
//*****************************************************************************
static int is_valid_ip(char *ip_str){
    int num, dots = 0;
    char *octetptr = NULL;
    char lstr[16];
    char* next_token = NULL;

	strcpy(lstr,ip_str);

    if (lstr == NULL)
        return 0;

    octetptr = strtok_r(lstr, DELIM, &next_token);

    if (octetptr == NULL)
        return 0;

    while (octetptr != NULL) {

        /* after parsing string, it must contain only digits */
        if (!valid_digit(octetptr))
            return 0;

        num = atoi(octetptr);

        /* check for valid IP */
        if (num >= 0 && num <= 255) {
            /* parse remaining string */
            octetptr = strtok_r(NULL, DELIM, &next_token);
            if (octetptr != NULL)
                ++dots;
        } else
            return 0;
    }

    /* valid IP string must contain 3 dots */
    if (dots != 3)
        return 0;
    return 1;
}

void PrintHelp()
{
	printf("Vaunix Attenuator Demonstration\n");
	printf("\n");
	printf("Hit CTRL+C to exit\n");
	printf("\n");

	printf(" --- Overall modes and device selection based on ip address ---\n");
	printf("  -d i 		Select the device to work with, i is the device ip address (192.168.100.11)\n");
	printf("  -r  		Read the current information of the device\n");
	printf("  -y		Save the current settings in the device.\n");
	printf("\n");
	printf("  -b		Batch mode, exit immediately after sending commands to the Lab Bricks.\n");
	printf("  -q		Quiet mode, skip most outputs.\n");
	printf("\n");

	printf(" --- Commands to set parameters and start ramp --- \n");
	printf("  -c n      Set the active channel\n");
	printf("  -f nn     Set working frequency, nn is working frequency in MHz\n");
	printf("  -a nn     Set attenuation, nn is attenuation in db units\n");
	printf("  -w nn     Set idle time between attenuator ramps, nn is time in ms.\n");
	printf("  -h nn     Set hold time between ramp phases\n");
	printf("  -s nn     Set ramp start value, nn is start value in db units\n");
	printf("  -e nn     Set ramp end value, nn is end value in db units\n");
	printf("  -t p nn   Set time to dwell on each attenuation value, nn is time in ms., p is ramp phase 1 or 2\n");

	printf("  -i p nn   Set attenuation ramp increment, nn is the increment\n");
	printf("            in db units. p is ramp phase 1 or 2\n");
	printf("  -g n      Start a ramp, 1 = once upwards, 2 = continuous upwards\n");
	printf("            5 = once down, 6 = continuous down, 17 = bidirectional once,\n");
	printf("            18 = continuous bidirectional ramps, 0 to stop\n");

	printf("  -x n rc rc ...	Start multiple ramps, n = number of ramps,  one rc per channel = \n");
	printf("					1 = once upwards, 2 = continuous upwards\n");
	printf("					5 = once down, 6 = continuous down, 17 = bidirectional once,\n");
	printf("					18 = continuous bidirectional ramps, 0 to stop\n");

	printf("\n");


}

// -------------------- - MakeLower------------------------------

MakeStrLower(char *str)
{
	int i;

	for(i = 0; i < strlen(str); i++)
	{
		str[i] = (char)tolower(str[i]);
	}
}

// remove the first character from the string
RemoveDash(char *str)
{
	int i;

	for(i = 0; i < strlen(str); i++)
	{
		str[i] = str[i+1];				// in effect we just shift the chars by one position
	}
}
	

// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should abort for another reason.
bool ParseCommandLine(int argc, char *argv[])
{
	int RampPhase;
	char devipstr[16];
	char thisParam[256];
	size_t icount;
	int i;

	enum {
		wantDash, wantIdle, wantAStart, wantAStop, wantDwell, wantAStep,
		wantAtten, wantSetRFOnOff, wantSweep, wantGetParam, wantDevID,
		wantDwell2, wantAStep2, wantHold, wantDwellPhase, wantStepPhase, wantWorkingFrequency,
		wantChannel, wantNumToRamp, wantRampCmd
	} state = wantDash;

	thisParam[255] = 0;	// safety backstop for errant strings


	for (i = 1; i < argc; ++i) {

		// Convert each argument to lowercase
		strncpy(thisParam, argv[i], 255);		// the limit is just to defend our string arrays

		MakeStrLower(thisParam);

		if (state == wantDash)
		{
			if ('-' != thisParam[0])
			{
				printf("\n *** Error in command line syntax *** \n");
				PrintHelp();
				return FALSE;
			}
			// remove the dash from the front of the string
			RemoveDash(thisParam);

			// Identify the command line arguments
			if ('d' == thisParam[0]) {
				state = wantDevID;
			}
			else if ('r' == thisParam[0]) {
				gbWantGetParam = TRUE;
				state = wantDash;
			}
			else if ('w' == thisParam[0]) {
				gbWantSetIdle  = TRUE;
				state = wantIdle;
			}
			else if ('s' == thisParam[0]) {
				gbWantSetAStart = TRUE;
				state = wantAStart;
			}
			else if ('e' == thisParam[0]) {
				gbWantSetAStop = TRUE;
				state = wantAStop;
			}
			else if ('t' == thisParam[0]) {
				state = wantDwellPhase;
			}
			else if ('i' == thisParam[0]) {
				state = wantStepPhase;
			}
			else if ('a' == thisParam[0]) {
				gbWantSetAttenuation = TRUE;
				state = wantAtten;
			}
			else if ('g' == thisParam[0]) {
				gbWantStartSweep = TRUE;
				state = wantSweep;
			}
			else if ('r' == thisParam[0]) {
				gbWantSetRFOnOff = TRUE;
				state = wantSetRFOnOff;
			}
			else if ('y' == thisParam[0]) {
				gbWantSaveSettings = TRUE;
				state = wantDash;
			}
			else if ('b' == thisParam[0]) {
				gbBatchMode = TRUE;
				state = wantDash;
			}
			else if ('q' == thisParam[0]) {
				gbQuietMode = TRUE;
				state = wantDash;
			}
			else if ('h' == thisParam[0]) {
				gbWantSetHold = TRUE;
				state = wantHold;
			}
			else if ('f' == thisParam[0]) {
				gbWantSetWorkingFrequency = TRUE;
				state = wantWorkingFrequency;
			}
			else if ('c' == thisParam[0]) {
				gbWantChannel = TRUE;
				state = wantChannel;
			}
			else if ('x' == thisParam[0]) {
				gbWantMCRamp = TRUE;
				NumToRamp = 0;
				iRampMode = 0;
				state = wantNumToRamp;
			}

			else {
				// this case is for "-?" and any argument we don't recognize
				PrintHelp();
				return FALSE;	// don't continue
			}
		}

		else {

			// save the whole substring and do conversions for each argument type
			switch (state){

			case wantDevID:
				if(is_valid_ip(thisParam))
				{
					strcpy(gldadevicesip, thisParam);
					gbDeviceOpen = TRUE;
				}
				else
					printf("Invalid IP Address\n");
				state = wantDash;
				break;

			case wantDwellPhase:
				RampPhase = atoi(thisParam);
				if (RampPhase == 1){
					gbWantSetDwell = TRUE;
					state = wantDwell;
				}
				else if (RampPhase == 2){
					gbWantSetDwell2 = TRUE;
					state = wantDwell2;
				}
				else state = wantDash;		// phase value is wrong, not much we can do about it...
				break;

			case wantStepPhase:
				RampPhase = atoi(thisParam);
				if (RampPhase == 1){
					gbWantSetAStep = TRUE;
					state = wantAStep;
				}
				else if (RampPhase == 2){
					gbWantSetAStep2 = TRUE;
					state = wantAStep2;
				}
				else state = wantDash;		// phase value is wrong, not much we can do about it...
				break;

			case wantIdle:
				IdleTime = atoi(thisParam);	// convert to a int
				state = wantDash;
				break;

			case wantHold:
				HoldTime = atoi(thisParam);
				state = wantDash;
				break;

			case wantChannel:
				Channel = atoi(thisParam);
				state = wantDash;
				break;

			case wantAStart:
				AStart = (float)atof(thisParam);
				state = wantDash;
				break;

			case wantAStop:
				AStop = (float)atof(thisParam);
				state = wantDash;
				break;

			case wantDwell:
				Dwell = atoi(thisParam);
				state = wantDash;
				break;

			case wantDwell2:
				Dwell2 = atoi(thisParam);
				state = wantDash;
				break;

			case wantAStep:
				AStep =  (float)atof(thisParam);
				state = wantDash;
				break;

			case wantAStep2:
				AStep2 =  (float)atof(thisParam);
				state = wantDash;
				break;

			case wantAtten:
				Attenuation = (float)atof(thisParam);	// cast to a float, atof actually returns a double
				state = wantDash;
				break;

			case wantWorkingFrequency:
				WorkingFrequency = (int)atof(thisParam); // cast to an int, atof actually returns a double
				state = wantDash;
				break;

			case wantSetRFOnOff:
				RFOnOff = atoi(thisParam);
				state = wantDash;
				break;

			case wantSweep:
				Sweep_mode = atoi(thisParam);
				state = wantDash;
				break;

			case wantNumToRamp:
				NumToRamp = atoi(thisParam);
				if (NumToRamp < 0) NumToRamp = 0;
				if (NumToRamp > 8) NumToRamp = 8;
				state = wantRampCmd;
				break;

			case wantRampCmd:
				RampMode[iRampMode] = atoi(thisParam);
				iRampMode++;
				if (iRampMode < NumToRamp) state = wantRampCmd;
				else state = wantDash;
				break;

			}
		}
	}

	if (state != wantDash) {
		// we are expecting an argument, if we didn't get one then print the help message
		PrintHelp();
		return FALSE;
	}

	// It's OK to continue
	return TRUE;
}


int main(int argc, char *argv[])
{
    bool realhardware;
    unsigned int index=0;
    float fdata;
	bool bTemp;
    LDADEVICE_DATA_T  ldadevicedata[LDA_DEVICES]; // MAX DEVICES list

	if (!ParseCommandLine(argc, argv))
		return 0;

	// LDA Data Structure Initialization
    fnLDA_Init();
	
    /* If you have actual hardware attached, set this to TRUE. Setting to FALSE will run in test mode */
    realhardware = TRUE;
    fnLDA_SetTestMode(!realhardware);

	// Initialize the LDA Device
	if(gbDeviceOpen)
	{
	    if (fnLDA_InitDevice((char *)gldadevicesip))
	        printf("Device Connection Error:%s\n", gldadevicesip);
		else
			printf("Device Connection Success:%s\n", gldadevicesip);		
	}

	if(gbWantGetParam)
	{
		if (fnLDA_CheckDeviceReady((char *)gldadevicesip) == STATUS_OK)
        {
        	// Device IP
            printf("Device IP:%s\n", gldadevicesip);
            printf("----------------------------------------------\n");

            // Device Name
            fnLDA_GetModelName((char *)gldadevicesip, ldadevicedata[index].modelname);
            printf("Model Name:%s\n", ldadevicedata[index].modelname);

            // Serial#
            fnLDA_GetSerialNumber((char *)gldadevicesip, &ldadevicedata[index].serialnumber);
            printf("Serial Number:%d\n", ldadevicedata[index].serialnumber);

            // Sw Version #
            fnLDA_GetSoftwareVersion((char *)gldadevicesip, ldadevicedata[index].swversion);
            printf("SW Version:%s\n", ldadevicedata[index].swversion);

            // IP Mode
            fnLDA_GetIPMode((char *)gldadevicesip, &ldadevicedata[index].ipmode);
            printf("IP Mode:%d\n", ldadevicedata[index].ipmode);

            // IP Address
            fnLDA_GetIPAddress((char *)gldadevicesip, ldadevicedata[index].ipaddress);
            printf("IP Address:%s\n", ldadevicedata[index].ipaddress);

            // Netmask
            fnLDA_GetNetmask((char *)gldadevicesip, ldadevicedata[index].netmask);
            printf("Subnet Mask:%s\n", ldadevicedata[index].netmask);

            // Gateway
            fnLDA_GetGateway((char *)gldadevicesip, (char *)ldadevicedata[index].gateway);
            printf("Gateway:%s\n", ldadevicedata[index].gateway);

            // Min Frequency
            fnLDA_GetMinWorkingFrequency((char *)gldadevicesip, &ldadevicedata[index].minfrequency);
            printf("Min Frequency:%d\n", (ldadevicedata[index].minfrequency / 10));

            // Max Frequency
            fnLDA_GetMaxWorkingFrequency((char *)gldadevicesip, &ldadevicedata[index].maxfrequency);
            printf("Min Frequency:%d\n", (ldadevicedata[index].maxfrequency / 10));

            // Min Attenuation
            fnLDA_GetMinAttenuation((char *)gldadevicesip, &ldadevicedata[index].minattenuation);
            fdata = (float)(ldadevicedata[index].minattenuation / 20.0);
            printf("Min Attenuation:%.2f\n", fdata);

            // Max Attenuation
            fnLDA_GetMaxAttenuation((char *)gldadevicesip, &ldadevicedata[index].maxattenuation);
            fdata = (float)(ldadevicedata[index].maxattenuation/20.0);
            printf("Max Attenuation:%.2f\n",fdata);

            // Channel #
            fnLDA_GetChannel((char *)gldadevicesip, &ldadevicedata[index].rf_channel);
            printf("Channel:%d\n",ldadevicedata[index].rf_channel);

            // Current Frequency
            fnLDA_GetWorkingFrequency((char *)gldadevicesip, &ldadevicedata[index].rf_current_frequency);
            printf("Frequency:%d\n",(ldadevicedata[index].rf_current_frequency/10));

            // Current Attenuation
            fnLDA_GetAttenuation((char *)gldadevicesip, &ldadevicedata[index].rf_attenuation);
            fdata = (float)(ldadevicedata[index].rf_attenuation/20.0);
            printf("Attenuation:%.2f\n",fdata);

            // Ramp Start
            fnLDA_GetRampStart((char *)gldadevicesip, &ldadevicedata[index].rampstart_attenuation);
            fdata = (float)(ldadevicedata[index].rampstart_attenuation/20.0);
            printf("Ramp Start:%.2f\n",fdata);

            // Ramp End
            fnLDA_GetRampEnd((char *)gldadevicesip, &ldadevicedata[index].rampstop_attenuation);
            fdata = (float)(ldadevicedata[index].rampstop_attenuation/20.0);
            printf("Ramp Stop:%.2f\n",fdata);

            // Dwell Time
            fnLDA_GetDwellTime((char *)gldadevicesip, &ldadevicedata[index].ramp_dwelltime);
            printf("Dwell Time:%d\n",(ldadevicedata[index].ramp_dwelltime));

            // Dwell Time2
            fnLDA_GetDwellTimeTwo((char *)gldadevicesip, &ldadevicedata[index].ramp_bidirectional_dwelltime);
            printf("Bi-directional Dwell Time:%d\n",(ldadevicedata[index].ramp_bidirectional_dwelltime));

            // Idle Time
            fnLDA_GetIdleTime((char *)gldadevicesip, &ldadevicedata[index].ramp_idletime);
            printf("Idle Time:%d\n",(ldadevicedata[index].ramp_idletime));

            // Hold Time
            fnLDA_GetHoldTime((char *)gldadevicesip, &ldadevicedata[index].ramp_bidirectional_holdtime);
            printf("Hold Time:%d\n",(ldadevicedata[index].ramp_bidirectional_holdtime));

            // Get Profile Count
            fnLDA_GetProfileCount((char *)gldadevicesip, &ldadevicedata[index].profile_count);
            printf("Profile Count:%d\n",(ldadevicedata[index].profile_count));

            // Get Profile Max Length
            fnLDA_GetProfileMaxLength((char *)gldadevicesip, &ldadevicedata[index].profile_maxlength);
            printf("Profile Maxlength:%d\n",(ldadevicedata[index].profile_maxlength));

            // Get Profile Dwell Time
            fnLDA_GetProfileDwellTime((char *)gldadevicesip, &ldadevicedata[index].profile_dwelltime);
            printf("Profile Dwell Time:%d\n",(ldadevicedata[index].profile_dwelltime));

            // Get Profile Idle Time
            fnLDA_GetProfileIdleTime((char *)gldadevicesip, &ldadevicedata[index].profile_idletime);
            printf("Profile Idle Time:%d\n",(ldadevicedata[index].profile_idletime));

        }
		gbWantGetParam = FALSE;
    }

	//Set channel
	if (gbWantChannel)
	{
		if (!gbQuietMode) printf("Setting the channel for device %s to %d", gldadevicesip, Channel);
		fnLDA_SetChannel((char *)gldadevicesip, Channel);
	}

	// Set Frequency
	if(gbWantSetWorkingFrequency)
	{
		if (!gbQuietMode) printf("Working Frequency set to  = %d Mhz\n", WorkingFrequency);
		fnLDA_SetWorkingFrequency((char *)gldadevicesip, WorkingFrequency*10);
	}

	// Set Attenuation
	if(gbWantSetAttenuation)
	{
		// using the HiRes API function with .05db units
		ScaledAttenuation = (int)(Attenuation * 20);
		
		// Set the selected channel with the attenuation
		if (!gbQuietMode) printf("Setting the attenuation for device %s to %.2f db\n", gldadevicesip, Attenuation);
		fnLDA_SetAttenuation((char *)gldadevicesip, ScaledAttenuation);
	}

	
	// Idle time
	if(gbWantSetIdle)
	{
		if (!gbQuietMode) printf("Setting the idle time between ramps for device %s to %d ms. \n", gldadevicesip, IdleTime);
		fnLDA_SetIdleTime((char *)gldadevicesip, IdleTime);
	}

	// Hold time
	if(gbWantSetHold)
	{
		if (!gbQuietMode) printf("Setting the hold time between ramp phases for device %s to %d ms \n", gldadevicesip, HoldTime);
		fnLDA_SetHoldTime((char *)gldadevicesip, HoldTime);
	}

	// DWell time
	if (gbWantSetDwell){
	
		if (!gbQuietMode) printf("Setting the first phase dwell time for device %s to %d \n", gldadevicesip, Dwell);
		fnLDA_SetDwellTime((char *)gldadevicesip, Dwell);
	}

	// Dwell time2
	if (gbWantSetDwell2)
	{
		if (!gbQuietMode) printf("Setting the second phase dwell time for device %s to %d ms \n", gldadevicesip, Dwell2);
		fnLDA_SetDwellTimeTwo((char *)gldadevicesip, Dwell2);
	}

	// Ramp Start
	if (gbWantSetAStart){
	
		if (!gbQuietMode) printf("Setting the ramp start for device %s to %.2f db \n", gldadevicesip, AStart);
		fnLDA_SetRampStart((char *)gldadevicesip, (int)(AStart*20.0));
	}

	// Ramp Stop
	if (gbWantSetAStop){
	
		if (!gbQuietMode) printf("Setting ramp end for device %s to %.2f db \n", gldadevicesip, AStop);
		fnLDA_SetRampEnd((char *)gldadevicesip, (int)(AStop*20.0));
	}


	if (gbWantSetAStep){
		if (!gbQuietMode) printf("Setting the first phase attenuation step for device %s to %.2f db \n",gldadevicesip, AStep);
		fnLDA_SetAttenuationStep((char *)gldadevicesip,(int)(AStep*20.0));
	}


	if (gbWantSetAStep2){
		if (!gbQuietMode) printf("Setting the first phase attenuation step2 for device %s to %.2f db \n",gldadevicesip, AStep2);
		fnLDA_SetAttenuationStep((char *)gldadevicesip,(int)(AStep2*20.0));
	}

	// Ramp Mode and Direction
	if (gbWantStartSweep)
	{
		if (Sweep_mode == 0)
		{
			if (!gbQuietMode) printf("Stopping the Attenuation Ramp\n");
			fnLDA_StartRamp((char *)gldadevicesip, FALSE);
			
		}
		else
		{
	
			// --- The user wants to start some kind of an attenuation ramp ---
			if (Sweep_mode & CL_SWP_DIRECTION)
			{
				bTemp = FALSE;
			}
			else
			{
				bTemp = TRUE;
			}	// NB -- the flag is TRUE for "up" in the Set...Direction call.
			// but the old test program uses a 0 bit for up, and a 1 bit for down...
	
			fnLDA_SetRampDirection((char *)gldadevicesip, bTemp);
	
			// --- and now we'll do the mode - one time or repeated ---
			if (Sweep_mode & CL_SWP_ONCE)
			{
				bTemp = FALSE;
			}
			else
			{
				bTemp = TRUE;
			}	// NB -- the flag is TRUE for "repeated" in the SetSweepMode call.
			// but the old test program encodes the modes differently
	
			fnLDA_SetRampMode((char *)gldadevicesip, bTemp);
	
			if (Sweep_mode & CL_SWP_BIDIRECTIONALLY)
			{
				bTemp = TRUE;
			}							// the command line has true for bidirectional 
			else						// as does the actual HW command...
			{
				bTemp = FALSE;
			}
	
			printf("Bidirection mode set to %x \n", bTemp);
			fnLDA_SetRampBidirectional((char *)gldadevicesip, bTemp);
			
			if (!gbQuietMode) printf("Starting an attenuation ramp for device %s\n",gldadevicesip);
			fnLDA_StartRamp((char *)gldadevicesip, TRUE);

		}
	}
	
	// Save settings
	if (gbWantSaveSettings)
	{
		if (!gbQuietMode) printf("Saving the settings for device %s\n", gldadevicesip);
		fnLDA_SaveSettings((char *)gldadevicesip);
	}


	// Close the device socket at the end of the process
	if(gbDeviceOpen)
	{
		fnLDA_CloseDevice((char *)gldadevicesip);
		gbDeviceOpen = FALSE;
	}

   
}

// Run program: Ctrl + F5 or Debug > Start Without Debugging menu
// Debug program: F5 or Debug > Start Debugging menu

// Tips for Getting Started: 
//   1. Use the Solution Explorer window to add/manage files
//   2. Use the Team Explorer window to connect to source control
//   3. Use the Output window to see build output and other messages
//   4. Use the Error List window to view errors
//   5. Go to Project > Add New Item to create new code files, or Project > Add Existing Item to add existing code files to the project
//   6. In the future, to open this project again, go to File > Open > Project and select the .sln file
